// 
// MenuItemBackend.cs
//  
// Author:
//       Lluis Sanchez <lluis@xamarin.com>
// 
// Copyright (c) 2011 Xamarin Inc
// 
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
// 
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.

using System;
using Xwt.Backends;
using Xwt.Drawing;
using System.Collections.Generic;


namespace Xwt.GtkBackend
{
	public class MenuItemBackend: IMenuItemBackend
	{
		IMenuItemEventSink eventSink;
		Gtk.MenuItem item;
		Gtk.Label label;
		List<MenuItemEvent> enabledEvents;
		bool changingCheck;
		ApplicationContext context;
		
		public MenuItemBackend ()
			: this (new Gtk.ImageMenuItem (""))
		{
		}
		
		public MenuItemBackend (Gtk.MenuItem item)
		{
			this.item = item;
			label = (Gtk.Label) item.Child;
			item.ShowAll ();
		}
		
		public Gtk.MenuItem MenuItem {
			get { return item; }
		}

		public void Initialize (IMenuItemEventSink eventSink)
		{
			this.eventSink = eventSink;
		}

		public void SetSubmenu (IMenuBackend menu)
		{
			if (menu == null)
				item.Submenu = null;
			else {
				Gtk.Menu m = ((MenuBackend)menu).Menu;
				item.Submenu = m;
			}
		}

		ImageDescription? defImage, selImage;

		public void SetImage (ImageDescription image)
		{
			Gtk.ImageMenuItem it = item as Gtk.ImageMenuItem;
			if (it == null)
				return;
			if (!image.IsNull) {
				if (defImage == null)
					item.StateChanged += ImageMenuItemStateChanged;
				defImage = image;
				selImage = new ImageDescription {
					Backend = image.Backend,
					Size = image.Size,
					Alpha = image.Alpha,
					Styles = image.Styles.Add ("sel")
				};
				var img = new ImageBox (context, image);
				img.ShowAll ();
				it.Image = img;
				GtkWorkarounds.ForceImageOnMenuItem (it);
			} else {
				if (defImage.HasValue) {
					item.StateChanged -= ImageMenuItemStateChanged;
					defImage = selImage = null;
				}
				it.Image = null;
			}
		}

		void ImageMenuItemStateChanged (object o, Gtk.StateChangedArgs args)
		{
			var it = item as Gtk.ImageMenuItem;
			var image = it?.Image as ImageBox;
			if (image == null || selImage == null || defImage == null)
				return;
			if (it.State == Gtk.StateType.Prelight)
				image.Image = selImage.Value;
			else if (args.PreviousState == Gtk.StateType.Prelight)
				image.Image = defImage.Value;
		}

		public string Label {
			get {
				return label != null ? (label.UseUnderline ? label.LabelProp : label.Text) : "";
			}
			set {
				if (formattedText != null) {
					formattedText = null;
					label.ApplyFormattedText (null);
				}
				if (label.UseUnderline)
					label.TextWithMnemonic = value;
				else
					label.Text = value;
			}
		}

		public string TooltipText {
			get {
				return item.TooltipText;
			}
			set {
				item.TooltipText = value;
			}
		}

		public bool UseMnemonic {
			get { return label.UseUnderline; }
			set { label.UseUnderline = value; }
		}
		
		public bool Sensitive {
			get {
				return item.Sensitive;
			}
			set {
				item.Sensitive = value;
			}
		}
		
		public bool Visible {
			get {
				return item.Visible;
			}
			set {
				item.Visible = value;
			}
		}
		
		public bool Checked {
			get { return (item is Gtk.CheckMenuItem) && ((Gtk.CheckMenuItem)item).Active; }
			set {
				if (item is Gtk.CheckMenuItem) {
					changingCheck = true;
					((Gtk.CheckMenuItem)item).Active = value;
					changingCheck = false;
				}
			}
		}

		FormattedText formattedText;
		public void SetFormattedText (FormattedText text)
		{
			label.Text = text?.Text;
			formattedText = text;
			label.Realized -= HandleStyleUpdate;
			label.StyleSet -= HandleStyleUpdate;
			label.ApplyFormattedText(text);
			label.Realized += HandleStyleUpdate;
			label.StyleSet += HandleStyleUpdate;

		}

		void HandleStyleUpdate (object sender, EventArgs e)
		{
			// force text update with updated link color
			if (label.IsRealized && formattedText != null) {
				label.ApplyFormattedText (formattedText);
			}
		}

/*		public void SetType (MenuItemType type)
		{
			string text = label.Text;
			
			Gtk.MenuItem newItem = null;
			switch (type) {
			case MenuItemType.Normal:
				if (!(item is Gtk.ImageMenuItem))
					newItem = new Gtk.ImageMenuItem (text);
				break;
			case MenuItemType.CheckBox:
				if (item.GetType () != typeof(Gtk.CheckMenuItem))
					newItem = new Gtk.CheckMenuItem (text);
				break;
			case MenuItemType.RadioButton:
				if (!(item is Gtk.RadioMenuItem))
					newItem = new Gtk.RadioMenuItem (text);
				break;
			}
			
			if (newItem != null) {
				if ((newItem is Gtk.CheckMenuItem) && (item is Gtk.CheckMenuItem))
					((Gtk.CheckMenuItem)item).Active = ((Gtk.CheckMenuItem)newItem).Active;
				newItem.Sensitive = item.Sensitive;
				if (item.Parent != null) {
					Gtk.Menu m = (Gtk.Menu)item.Parent;
					int pos = Array.IndexOf (m.Children, item);
					m.Insert (newItem, pos);
					m.Remove (item);
				}
				newItem.ShowAll ();
				if (!item.Visible)
					newItem.Hide ();
				
				if (enabledEvents != null) {
					foreach (var ob in enabledEvents)
						DisableEvent (ob);
				}
				
				item = newItem;
				label = (Gtk.Label) item.Child;
				
				if (enabledEvents != null) {
					foreach (var ob in enabledEvents)
						EnableEvent (ob);
				}
			}
		}*/
		
		public void InitializeBackend (object frontend, ApplicationContext context)
		{
			this.context = context;
		}

		public void EnableEvent (object eventId)
		{
			if (eventId is MenuItemEvent) {
				if (enabledEvents == null)
					enabledEvents = new List<MenuItemEvent> ();
				enabledEvents.Add ((MenuItemEvent)eventId);
				if ((MenuItemEvent)eventId == MenuItemEvent.Clicked)
					item.Activated += HandleItemActivated;
			}
		}

		public void DisableEvent (object eventId)
		{
			if (eventId is MenuItemEvent) {
				enabledEvents.Remove ((MenuItemEvent)eventId);
				if ((MenuItemEvent)eventId == MenuItemEvent.Clicked)
					item.Activated -= HandleItemActivated;
			}
		}

		void HandleItemActivated (object sender, EventArgs e)
		{
			if (!changingCheck) {
				context.InvokeUserCode (eventSink.OnClicked);
			}
		}

		public void Dispose ()
		{
			if (label != null) {
				label.Realized -= HandleStyleUpdate;
				label.StyleSet -= HandleStyleUpdate;
				label = null;
			}
		}
	}
}

