//
// Copyright (c) 2012-2021 Antmicro
//
// This file is licensed under the MIT License.
// Full license text is available in the LICENSE file.

using System;
using Antmicro.Migrant.Utilities;
using System.Reflection;
using System.Linq;

namespace Antmicro.Migrant.VersionTolerance
{
    internal class MethodDescriptor : IIdentifiedElement
    {
        public MethodDescriptor()
        {
        }

        public MethodDescriptor(MethodInfo method)
        {
            UnderlyingMethod = method;
        }

        public void Write(ObjectWriter writer)
        {
            writer.TouchAndWriteTypeId(UnderlyingMethod.ReflectedType);

            var methodParameters = UnderlyingMethod.GetParameters();
            if(UnderlyingMethod.IsGenericMethod)
            {
                var genericDefinition = UnderlyingMethod.GetGenericMethodDefinition();
                var genericArguments = UnderlyingMethod.GetGenericArguments();
                var genericMethodParamters = genericDefinition.GetParameters();

                writer.PrimitiveWriter.Write(genericDefinition.Name);
                writer.PrimitiveWriter.Write(genericArguments.Length);
                for(int i = 0; i < genericArguments.Length; i++)
                {
                    writer.TouchAndWriteTypeId(genericArguments[i]);
                }

                writer.PrimitiveWriter.Write(genericMethodParamters.Length);
                for(int i = 0; i < genericMethodParamters.Length; i++)
                {
                    writer.PrimitiveWriter.Write(genericMethodParamters[i].ParameterType.IsGenericParameter);
                    if(genericMethodParamters[i].ParameterType.IsGenericParameter)
                    {
                        writer.PrimitiveWriter.Write(genericMethodParamters[i].ParameterType.GenericParameterPosition);
                    }
                    else
                    {
                        writer.TouchAndWriteTypeId(methodParameters[i].ParameterType);
                    }
                }
            }
            else
            {
                writer.PrimitiveWriter.Write(UnderlyingMethod.Name);
                writer.PrimitiveWriter.Write(0); // no generic arguments
                writer.PrimitiveWriter.Write(methodParameters.Length);

                foreach(var p in methodParameters)
                {
                    writer.TouchAndWriteTypeId(p.ParameterType);
                }
            }
        }

        public void Read(ObjectReader reader)
        {
            var type = reader.ReadType().UnderlyingType;
            var methodName = reader.PrimitiveReader.ReadString();
            var genericArgumentsCount = reader.PrimitiveReader.ReadInt32();
            var genericArguments = new Type[genericArgumentsCount];
            for(int i = 0; i < genericArgumentsCount; i++)
            {
                genericArguments[i] = reader.ReadType().UnderlyingType;
            }

            var parametersCount = reader.PrimitiveReader.ReadInt32();
            if(genericArgumentsCount > 0)
            {
                var parameters = new TypeOrGenericTypeArgument[parametersCount];
                for(int i = 0; i < parameters.Length; i++)
                {
                    var genericType = reader.PrimitiveReader.ReadBoolean();
                    parameters[i] = genericType ? 
                        new TypeOrGenericTypeArgument(reader.PrimitiveReader.ReadInt32()) :
                        new TypeOrGenericTypeArgument(reader.ReadType().UnderlyingType);
                }

                UnderlyingMethod = type.GetMethods(BindingFlags.Instance | BindingFlags.Static | BindingFlags.Public | BindingFlags.NonPublic).SingleOrDefault(m => 
                    m.IsGenericMethod && 
                    m.GetGenericMethodDefinition().Name == methodName && 
                    m.GetGenericArguments().Length == genericArgumentsCount && 
                    CompareGenericArguments(m.GetGenericMethodDefinition().GetParameters(), parameters));

                if(UnderlyingMethod != null)
                {
                    UnderlyingMethod = UnderlyingMethod.MakeGenericMethod(genericArguments);
                }
            }
            else
            {
                var types = new Type[parametersCount];
                for(int i = 0; i < types.Length; i++)
                {
                    types[i] = reader.ReadType().UnderlyingType;
                }

                UnderlyingMethod = type.GetMethod(methodName, BindingFlags.Instance | BindingFlags.Static | BindingFlags.Public | BindingFlags.NonPublic, null, types, null);
            }
        }

        public MethodInfo UnderlyingMethod { get; private set; }

        private static bool CompareGenericArguments(ParameterInfo[] actual, TypeOrGenericTypeArgument[] expected)
        {
            if(actual.Length != expected.Length)
            {
                return false;
            }

            for(int i = 0; i < actual.Length; i++)
            {
                if(actual[i].ParameterType.IsGenericParameter)
                {
                    if(actual[i].ParameterType.GenericParameterPosition != expected[i].GenericTypeArgumentIndex)
                    {
                        return false;
                    }
                }
                else
                {
                    if(actual[i].ParameterType != expected[i].Type)
                    {
                        return false;
                    }
                }
            }

            return true;
        }
    }
}

